### ctmcd: An R Package for Estimating the Parameters of a
### Continuous-Time Markov Chain From Discrete-Time Data

#install.packages("ctmcd")
#install.packages("expm")
#install.packages("foreach")
#install.packages("doParallel")

### Load Packages
library(ctmcd)
library(expm)

### Load Data
data(tm_abs)
tm_rel <- rbind((tm_abs/rowSums(tm_abs))[1:7,],c(rep(0,7),1))

### Matrix-Logarithm Adjustment Approaches
gmda <- gm(tm=tm_rel,te=1,method="DA")
gmwa <- gm(tm=tm_rel,te=1,method="WA")
gmqo <- gm(tm=tm_rel,te=1,method="QO")

plot(gmda)
plot(gmwa)
plot(gmqo)

### Expectation-Maximization Algorithm
gm0 <- matrix(1,8,8)
diag(gm0) <- 0
diag(gm0) <- -rowSums(gm0)
gm0[8,] <- 0

gmem <- gm(tm=tm_abs,te=1,method="EM",gmguess=gm0)
plot(gmem)
plot(gmem$ll,main="Expectation Maximization Algorithm\nLog Likelihood Path",
     xlab="Iteration",ylab="Log-Likelihood")

### Wald Confidence Interval
ciem <- gmci(gmem,alpha=.05) ### Takes approx. 5 s
plot(ciem)

### Gibbs Sampler
pr <- list()
pr[[1]] <- matrix(1,8,8)
pr[[1]][8,] <- 0
pr[[2]] <- c(rep(5,7),Inf)

gmgs <- gm(tm=tm_abs,te=1,method="GS",prior=pr,burnin=1000) ### Takes approx. 30 s
plot(gmgs)
plot(gmgs$l1metric,log="y",main="Gibbs Sampler\nL1 Metric of Subsequent
     Generator Matrix Estimates",xlab="Iteration", ylab="L1 Metric")

### Speedtest Sampling Algorithms
speedmat_modrej <- matrix(0,8,8)
speedmat_unif <- matrix(0,8,8)
tpm <- expm(gmgs$par)
for(i in 1:7){ ### Takes approx. 4 min
  for(j in 1:8){
    elem <- matrix(0,8,8)
    elem[i,j] <- 1e5
    t0 <- proc.time()
    rNijTRiT_ModRej(elem,1,gmgs$par)
    speedmat_modrej[i,j] <- (proc.time()-t0)[3]
    t0 <- proc.time()
    rNijTRiT_Unif(elem,1,gmgs$par,tpm)
    speedmat_unif[i,j] <- (proc.time()-t0)[3]
  }
}

plotM(speedmat_modrej,main="Time for Simulation of 100,000 Paths
Modified Rejection Sampling",xnames=rownames(tm_abs),ynames=colnames(tm_abs))
plotM(speedmat_unif,main="Time for Simulation of 100,000 Paths\n
      Uniformization Sampling",xnames=rownames(tm_abs),ynames=colnames(tm_abs))

### Parallelization
library(foreach)
library(doParallel)
N <- 1e5
nco <- detectCores()
cl <- makeCluster(nco)
registerDoParallel(cl) ### Takes approx. 5 min (8 Kernels)
gspar=foreach(i=1:nco,.combine="+",.packages=c("ctmcd","expm")) %dopar%
  gm(tm=tm_abs,te=1,method="GS",burnin=1000,prior=pr,eps=0,niter=N/nco)$par/nco
stopCluster(cl)

### Credibility Interval
cigs <- gmci(gm=gmgs,alpha=0.05)
plot(cigs)

### Discrete-Time Profile Curves/
### Transitions into Absorbing States
tmax <- 20
for(cat in 1:7){ ### Takes approx. 3 min
  absStvec <- sapply(1:tmax, function(t) expm(gmgs$par*t)[cat,8])
  quantMat <- matrix(0,4,tmax+1)
  for(t in 1:tmax){
    dtdraws <- lapply(gmgs$draws,function(x) expm(t*x))
    drawvec <- sapply(1:length(gmgs$draws),function(x) dtdraws[[x]][cat,8])
    quantMat[,t+1] <- quantile(drawvec,c(.025,.05,.95,.975))
  }
  plot(0:tmax,c(0,absStvec),t="l",lwd=3,ylim=c(0,max(quantMat)),
       main=paste0("Absorbing State Profiles\nInitial Rating Category ",
                   rownames(tm_abs)[cat]),xlab="Time [Years]",ylab="Probability of Default")
  for(i in 1:4)
    lines(0:tmax,quantMat[i,],lty=c(3,2,2,3)[i])
  legend("topleft",lty=c(3,2,1),c("95%","90%","Median"))
}