##################################################  
## Introduction

library(ade4)
library(adegraphics)



##################################################  
## A simple example

data("mafragh")
names(mafragh)
# [1] "xy"        "flo"       "neig"      "env"       "partition" "area"     
# [7] "tre"       "traits"    "nb"        "Spatial"   "spenames" 



##################################################  
## Basic elements and simple graphs
## Classes, objects and calling functions

pca1 <- dudi.pca(mafragh$env, scale = TRUE, center = TRUE, scannf = FALSE, nf = 4)

g_sc1 <- s.corcircle(pca1$co, plot = FALSE)
class(g_sc1)
# [1] "S2.corcircle"
# attr(,"package")
# [1] "adegraphics"

g_sl1 <- s.label(pca1$li)
class(g_sl1)
# [1] "S2.label"
# attr(,"package")
# [1] "adegraphics"

slotNames(g_sl1)
# [1] "data"         "trellis.par"  "adeg.par"     "lattice.call" "g.args"      
# [6] "stats"        "s.misc"       "Call"        

g_sl1@Call
# s.label(dfxy = pca1$li, labels = rownames(pca1$li), xax = 1, yax = 2, 
#         facets = NULL, plot = TRUE, storeData = TRUE, add = FALSE, pos = -1)



##################################################  
## Graphical parameters

g_sl2 <- s.label(pca1$li, plabels.optim = TRUE, paxes.draw = TRUE, plot = FALSE)
g_sl2@adeg.par$pbackground$col
# [1] "white"



##################################################  
## Manipulating a simple graph

col.labels <- rainbow(7)[as.numeric(mafragh$partition)]
update(g_sl2, ppoints.col = col.labels, pbackground.col = "grey90", pgrid.col = "white")
g_sl2@adeg.par$pbackground$col
# [1] "grey90"



##################################################
## Representation of spatial information

class(mafragh$Spatial)
# [1] "SpatialPolygons"
# attr(,"package")
# [1] "sp"

mycolors <- colorRampPalette(c("blue", "red"))(6)
g_sv1 <- s.value(mafragh$xy, z = pca1$li[, 1], Sp = mafragh$Spatial, method = "color", 
                 col = mycolors, pgrid.draw = FALSE, symbol = "circle", ppoints.cex = 0.5)



##################################################
## Dealing with multiple graphs
## Multivariate analysis outputs

pca2 <- dudi.pca(mafragh$flo, scale = FALSE, scannf = FALSE, nf = 3)
coi1 <- coinertia(pca1, pca2, scannf = FALSE, nf = 3)
g_coi1 <- plot(coi1)

length(g_coi1)
# [1] 6

names(g_coi1)
# [1] "Xax"       "Yax"       "eig"       "XYmatch"   "Yloadings" "Xloadings"

g_coi1$XYmatch
g_coi1[[4]]



##################################################
## Step-by-step creation
  
g_in1 <- insert(g_coi1$eig, g_coi1$Xloadings, posi = "topleft", ratio = 0.3)
class(g_in1)
# [1] "ADEgS"
# attr(,"package")
# [1] "adegraphics"
  
names(g_in1)
# [1] "g1" "g2"
names(g_in1) <- c("Env", "Eig")
names(g_in1)
# [1] "Env" "Eig"

g_in2 <- ADEgS(list(Env = g_coi1$Xloadings, Eig = g_coi1$eig), positions = rbind(c(0, 0, 1, 1), c(0, 0.7, 0.3, 1)))



##################################################
## Customizing an ADEgS

update(g_in2, Eig.pbackground.box = TRUE, Env.pgrid.draw = FALSE, psub.cex = 2)



##################################################
## Automatic collections

g_sc1 <- s.class(pca1$li, fac = mafragh$partition, col = TRUE, facets = mafragh$partition)

g_sv2 <- s.value(mafragh$xy, z = pca1$li, Sp = mafragh$Spatial, pgrid.draw = FALSE,
                 method = "color", symbol = "circle", col = mycolors, ppoints.cex = 0.4)

class(g_sv2)
# [1] "ADEgS"
# attr(,"package")
# [1] "adegraphics"

names(g_sv2)
# [1] "Axis1" "Axis2" "Axis3" "Axis4"

class(g_sv2$Axis1)
# [1] "S2.value"
# attr(,"package")
# [1] "adegraphics"



##################################################
## Conclusions
  
library(lattice)
mytext <- xyplot(0 ~ 0, axes = FALSE, 
                 panel = function(...) panel.text(0, 0, "adegraphics\n&\nlattice", cex = 4))

g_latt <- cbindADEg(g_sv2, mytext, plot = TRUE)
