#-----------------------------------------------------------------------------------------------------------
# Author: Lu Ou
# Date: 2018-01-22
# Filename: ou-hunter-chow.R
# Purpose: create a replication script for all results from 
#          the manuscript
# Usage:
# * Please consult the installation manual for installation guidance.
# * Please install the package from CRAN using
#   > install.packages("dynr")
# * Please also note that to successfully execute this script, the reader needs to have
#   1) a C compiler (e.g., gcc) and 2) the gsl library (https://www.gnu.org/software/gsl/)
#   installed on his/her system.
# * If you encounter any problems, please do not hesitate to email the author, Lu Ou ( lzo114@psu.edu )
#   for assistance.
#-----------------------------------------------------------------------------------------------------------


#---- (1) Load packages ----
require("dynr")

#------------------------------------------------------------------------------
# Example 1: Regime-Switching Linear State-Space Model
#------------------------------------------------------------------------------

#---- (2) Read in data and create dynr data object----
data("EMG")
EMGdata <- dynr.data(EMG, id = 'id', time = 'time', 
  observed = 'iEMG', covariates = 'SelfReport')

#---- (3) Specify recipes for all model pieces ----

#---- (3a) Measurement ----
recMeas <- prep.measurement(
  values.load = rep(list(matrix(1, 1, 1)), 2),
  values.int = list(matrix(4, 1, 1), matrix(3, 1, 1)),
  params.int = list(matrix('mu_1', 1, 1), matrix('mu_2', 1, 1)),
  values.exo = list(matrix(0, 1, 1), matrix(1, 1, 1)),
  params.exo = list(matrix('fixed', 1, 1), matrix('beta_2', 1, 1)),
  obs.names = c('iEMG'),
  state.names = c('eta'),
  exo.names = c("SelfReport"))

#---- (3b) Dynamic and measurement noise cov structures----

recNoise <- prep.noise(
  values.latent = matrix(1, 1, 1),
  params.latent = matrix('dynNoise', 1, 1),
  values.observed = matrix(0, 1, 1),
  params.observed = matrix('fixed', 1, 1))

# ---- (3c) Regimes-switching model ----

recReg <- prep.regimes(
  values = matrix(c(.7, -1, 0, 0), 2, 2),
  params = matrix(c('c11', 'c21', 'fixed', 'fixed'), 2, 2))

recReg2 <- prep.regimes(
  values = matrix(c(.8, -1, 0, 0), 2, 2),
  params = matrix(c('c_Delta11', 'c1', 'fixed', 'fixed'), 2, 2),
  deviation = TRUE, refRow = 2)

#---- (3d) Initial condition specification ----

recIni <- prep.initial(
  values.inistate = matrix(0, 1, 1),
  params.inistate = matrix('fixed', 1, 1),
  values.inicov = matrix(1, 1, 1),
  params.inicov = matrix('fixed', 1, 1),
  values.regimep = c(1, 0),
  params.regimep = c('fixed', 'fixed'))


#---- (3e) Dynamic model ----

recDyn <- prep.matrixDynamics(
  values.dyn = list(matrix(.1, 1, 1), matrix(.5, 1, 1)),
  params.dyn = list(matrix('phi_1', 1, 1), matrix('phi_2', 1, 1)),
  isContinuousTime = FALSE)

#---- (4a) Create model  ----

rsmod <- dynr.model(
  dynamics = recDyn,
  measurement = recMeas,
  noise = recNoise,
  initial = recIni,
  regimes = recReg,
  data = EMGdata,
  outfile = "RSLinearDiscrete.c")

#---- (4b) Check model specification  ----

printex(rsmod,
  ParameterAs = rsmod$param.names,
  printInit = TRUE, printRS = TRUE,
  outFile = "RSLinearDiscreteYang.tex")
#tools::texi2pdf("RSLinearDiscreteYang.tex")
#system(paste(getOption("pdfviewer"), "RSLinearDiscreteYang.pdf"))

#---- (4c) Create model and cook it all up  ----

yum <- dynr.cook(rsmod)

#---- (5) Serve it! ----

summary(yum)

plot(yum, dynrModel = rsmod, style = 1, textsize = 5)

pdf('./Figures/plotRSGG.pdf', height=7, width=12)
dynr.ggplot(yum, dynrModel = rsmod, style = 1,
  names.regime = c("Deactivated", "Activated"),
  title = "(B) Results from RS-AR model", numSubjDemo = 1,
  shape.values = NA, #c(1),
  text = element_text(size = 24), #24
  is.bw = TRUE)
dev.off()

autoplot(yum, dynrModel = rsmod, style = 1,
  names.regime = c("Deactivated", "Activated"),
  title = "(B) Results from RS-AR model", numSubjDemo = 1,
  shape.values = c(1),
  text = element_text(size = 16),
  is.bw = TRUE)

plotFormula(dynrModel = rsmod, ParameterAs = rsmod$param.names,
  printDyn = TRUE, printMeas = TRUE) +
  ggtitle("(A)") +
  theme(plot.title = element_text(hjust = 0.5, vjust = 0.01, size = 16)) 

plotFormula(dynrModel = rsmod, ParameterAs = coef(yum),
  printDyn = TRUE, printMeas = TRUE) +
  ggtitle("(B)") +
  theme(plot.title = element_text(hjust = 0.5, vjust = 0.01, size = 16)) 

#------------------------------------------------------------------------------
# Example 2.1: Nonlinear Continuous-time Models (Equations 20 & 21)
#------------------------------------------------------------------------------

# ---- Read in the data ----
data(PPsim)
PPdata <- dynr.data(PPsim, id = "id", time = "time", observed = c("x", "y"))

#---- Prepare the recipes (i.e., specifies modeling functions) ----

# Measurement (factor loadings)
meas <- prep.measurement(
  values.load = diag(1, 2),
  obs.names = c('x', 'y'),
  state.names = c('prey', 'predator'))

# alternatively, use prep.loadings
# meas <- prep.loadings(
#   map = list(
#     prey = "x",
#     predator = "y"),
#   params = NULL)

# Initial conditions on the latent state and covariance
initial <- prep.initial(
  values.inistate = c(3, 1),
  params.inistate = c("fixed", "fixed"),
  values.inicov = diag(c(0.01, 0.01)), 
  params.inicov = diag("fixed", 2)
)

#measurement and dynamics covariances
mdcov <- prep.noise(
  values.latent = diag(0, 2),
  params.latent = diag(c("fixed", "fixed"), 2),
  values.observed = diag(rep(0.3, 2)),
  params.observed = diag(c("var_1", "var_2"), 2)
)

# dynamics
preyFormula <- prey ~ a * prey - b * prey * predator
predFormula <- predator ~ - c * predator + d * prey * predator
ppFormula <- list(preyFormula, predFormula)
ppDynamics <- prep.formulaDynamics(formula = ppFormula,
  startval = c(a = 2.1, c = 0.8, b = 1.9, d = 1.1),
  isContinuousTime = TRUE)

#constraints
trans <- prep.tfun(formula.trans = list(a ~ exp(a), b ~ exp(b), 
                                        c ~ exp(c), d ~ exp(d)),
  formula.inv = list(a ~ log(a), b ~ log(b), 
                     c ~ log(c), d ~ log(d)))

#------------------------------------------------------------------------------
# Cooking materials

# Put all the recipes together in a Model Specification
model2.1 <- dynr.model(dynamics = ppDynamics, 
  measurement = meas, noise = mdcov, 
  initial = initial, transform = trans, 
  data = PPdata,
  outfile = "NonlinearODE.c")

printex(model2.1, 
  ParameterAs = model2.1$param.names,
  show = FALSE, printInit = TRUE,
  outFile = "NonlinearODE.tex")
#tools::texi2pdf("NonlinearODE.tex")
#system(paste(getOption("pdfviewer"), "NonlinearODE.pdf"))

# Estimate free parameters
res2.1 <- dynr.cook(dynrModel = model2.1)

# Examine results
# True parameter values a = 2, b = 2, c = 1, d = 1
summary(res2.1)
#       names parameters       s.e.  t-value  ci.lower  ci.upper 
# a         a  1.9637320 0.06946322 28.27010 1.8275866 2.0998774 
# c         c  1.0023304 0.03062620 32.72788 0.9423042 1.0623567 
# b         b  1.9327832 0.06216237 31.09250 1.8109472 2.0546192 
# d         d  0.9608279 0.02628627 36.55246 0.9093078 1.0123481 
# var_1 var_1  0.2399578 0.01089095 22.03277 0.2186119 0.2613036 
# var_2 var_2  0.2380317 0.01072899 22.18585 0.2170033 0.2590601 
# 
# -2 log-likelihood value at convergence = 2843.19
# AIC = 2855.19
# BIC = 2884.64

#------------------------------------------------------------------------------
# Example 2.2: Regime-Switching Nonlinear Continuous-time Model
#------------------------------------------------------------------------------
# ---- Read in the data ----
data("RSPPsim")
data <- dynr.data(RSPPsim, id = "id", time = "time",
  observed = c("x", "y"), covariate = "cond")

# ---- Prepare the recipes (i.e., specifies modeling functions) ----

# Measurement (factor loadings)
meas <- prep.measurement(
  values.load = diag(1, 2),
  obs.names = c('x', 'y'),
  state.names = c('prey', 'predator'))

# Initial conditions on the latent state and covariance
initial <- prep.initial(
  values.inistate = c(3, 1),
  params.inistate = c("fixed", "fixed"),
  values.inicov = diag(c(0.01, 0.01)), 
  params.inicov = diag("fixed", 2),
  values.regimep = c(.7, .3),
  params.regimep = c("fixed", "fixed"))

# Regime-switching function
# The RS model assumes that each element of the transition probability 
# matrix (TPM) can be expressed as a linear predictor (lp).
# LPM = 
# lp(p11) ~ 1 + x1 + x2 + ... + xn,   lp(p12) ~ 1 + x1 + x2 + ... + xn
# lp(p21) ~ 1 + x1 + x2 + ... + xn,   lp(p22) ~ 1 + x1 + x2 + ... + xn
# Here I am specifying lp(p12) and lp(p22); the remaining elements
# lp(p11) and lp(p21) are fixed at zero.
# nrow = numRegimes, ncol = numRegimes*(numCovariates+1)

regimes <- prep.regimes(
  values = matrix(c(0, 0, -1, 1.5,
                    0, 0, -1, 1.5),
                nrow = 2, ncol = 4, byrow = T), 
  params = matrix(c("fixed", "fixed", "int_1", "slp_1",
                    "fixed", "fixed", "int_2", "slp_2"), 
                nrow = 2, ncol = 4, byrow = T), 
  covariates = "cond")

#measurement and dynamics covariances
mdcov <- prep.noise(
  values.latent = diag(0, 2),
  params.latent = diag(c("fixed","fixed"), 2),
  values.observed = diag(rep(0.5,2)),
  params.observed = diag(rep("var_epsilon",2),2)
)

# dynamics
preyFormula <- prey ~ a * prey - b * prey * predator
predFormula <- predator ~ - c * predator + d * prey * predator
ppFormula <- list(preyFormula, predFormula)
cPreyFormula <- prey ~ a * prey - e * prey ^ 2 - b * prey * predator
cPredFormula <- predator ~ 
  f * predator - c * predator ^ 2 + d * prey * predator
cpFormula <- list(cPreyFormula, cPredFormula)
rsFormula <- list(ppFormula, cpFormula)

dynm <- prep.formulaDynamics(formula = rsFormula,
  startval = c(a = 2.1, c = 3, b = 1.2, d = 1.2, e = 1, f = 2),
  isContinuousTime = TRUE)

#constraints
trans<-prep.tfun(
  formula.trans = list(a ~ exp(a), b ~ exp(b), c ~ exp(c), 
                       d ~ exp(d), e ~ exp(e), f ~ exp(f)),
  formula.inv = list(a ~ log(a), b ~ log(b), c ~ log(c), 
                     d ~ log(d), e ~ log(e), f ~ log(f)))

#------------------------------------------------------------------------------
# Cooking materials

# Put all the recipes together in a Model Specification
model2.2 <- dynr.model(dynamics = dynm, measurement = meas,
  noise = mdcov, initial = initial,
  regimes = regimes, transform = trans,
  data = data,
  outfile = "RSNonlinearODE_1.c")

printex(model2.2, ParameterAs = model2.2$param.names, printInit = TRUE, printRS = TRUE,
  outFile = "RSNonlinearODE_1.tex")
#tools::texi2pdf("RSNonlinearODE_1.tex")
#system(paste(getOption("pdfviewer"), "RSNonlinearODE_1.pdf"))

model2.2$ub[ c("int_1", "int_2", "slp_1", "slp_2") ] <- c(0, 0, 10, 10)
model2.2$lb[ c("int_1", "int_2", "slp_1", "slp_2") ] <- c(-10, -10, 0, 0)

# Estimate free parameters
res2.2 <- dynr.cook(model2.2)

# Examine results
summary(res2.2)


plotFormula(model2.2, ParameterAs = model2.2$param.names) +
  ggtitle("(A)") +
  theme(plot.title = element_text(hjust = 0.5, vjust = 0.01, size = 16)) 

plotFormula(model2.2, ParameterAs = coef(res2.2)) +
  ggtitle("(B)") +
  theme(plot.title = element_text(hjust = 0.5, vjust = 0.01, size = 16))


dynr.ggplot(res2.2, model2.2, style = 1,
  names.regime = c("Summer", "Winter"),
  title = "", idtoPlot = 11,
  text = element_text(size = 16))
#ggsave("RSNonlinearODEggPlot1.pdf")

dynr.ggplot(res2.2, model2.2, style = 2, 
  names.regime = c("Summer", "Winter"),
  title = "", idtoPlot = 9,
  text = element_text(size = 16))
#ggsave("RSNonlinearODEggPlot2.pdf")

plot(res2.2, dynrModel = model2.2, style = 1)
plot(res2.2, dynrModel = model2.2, style = 2)
#------------------------------------------------------------------------------

# ds <- read.table('258_3H_NoOutlier.dat')
# names(ds) <- c('iEMG', 'V2', 'SelfReport', 'V4', 'V5')
# ds$time <- seq(0, by=0.2, length.out=nrow(ds))
# ds$id <- 1
# 
# dd <- dynr.data(ds, id='id', time='time', observed='iEMG', covariates='SelfReport')

pdf('plotEMG1Subj.pdf', height=7/1.5, width=12/1.5)
#pdf('plotEMG1SubjModel1.pdf', height=7/1.5, width=12/1.5)
#  zones=matrix(c(2,0,1,3), ncol=2, byrow=TRUE)
#  layout(zones, widths=c(4/5,1/5), heights=c(1/5,4/5))
ds = EMG
par(mar=c(4, 4, 2, 4) + 0.1, ps=12)
cexsize = 24/12
plot(NA, xlim=c(0,138.8), ylim=c(0, 12), xlab='',
     ylab='', xaxs="i", yaxs="i",
     las=1, xaxt ='n', yaxt='n', main="(A)", cex.main = cexsize)
bestRegime <- apply(yum@pr_t_given_T, 2, which.max)
xleft <- ds$time[-nrow(ds)]
xright <- ds$time[-1]
ybottom <- -1
ytop <- 15
#rect(xleft,ybottom,xright,ytop,col=c("#edaac1","#a7dab0")[bestRegime], border=NA)
#legend('topleft',
#	legend=c('Measured iEMG', 'Measured Self-Report', 'Estimated Activated Regime', 'Estimated Deactivated Regime'), lty=c(1, 4, 1, 1), col=c('black', 'black', "#edaac1", "#a7dab0"),
#	lwd=c(2, 2, 15, 15), bg='white')
legend('topleft',
       legend=c('Measured iEMG', 'Measured Self-Report'), lty=c(1, 4), col=c('black', 'black'),
       lwd=c(2, 2), bg='white', bty='n')
axis(side=4, at=seq(0, 12, by=2), las=1)
axis(side=1, at=seq(0, 100, by=50), las=1)#
axis(side=2, at=seq(-2, 6, by=2), las=1)#axis(side=2, at=seq(0, 15, by=5), las=1)
mtext('Self-Report', side=4, line=2, cex = cexsize)
mtext(expression(paste("Integrated  EMG  ( ", mu, "V)")), side=2, line=2, cex = cexsize)
mtext('Time (seconds)', side=1, line=2, cex = cexsize)
lines(ds$time, ds$iEMG, lty=1, col='black')
lines(ds$time, ds$SelfReport, lty='dotdash', lwd=2)
#  xhist <- hist(ds$time, plot=FALSE)
#  yhist <- hist(ds$iEMG, plot=FALSE)
#  top = max(c(xhist$counts, yhist$counts))
#  par(mar=c(0,3,1,1))
#  #barplot(xhist$counts, axes=FALSE, ylim=c(0, top), space=0)
#  barplot(NA, axes=FALSE, ylim=c(0, top), space=0)
#  par(mar=c(3,0,1,1))
#  barplot(yhist$counts, axes=FALSE, xlim=c(0, top), ylim=range(yhist$breaks), space=0, horiz=TRUE)
#  par(oma=c(3,3,0,0))

dev.off()

