# ----------------------- #
#  Install and load SIRE  #
# ----------------------- #
install.packages("SIRE")
library(SIRE)
library(igraph)
library(systemfit)

# ------------------------------- #
#  Simulated system of equations  #
#  Figures 2-3                    #
# ------------------------------- #

eq.system <- list(
  eq1 <- y1 ~ y5 + y7, eq2 <- y2 ~ z,
  eq3 <- y3 ~ y11, eq4 <- y4 ~ y3,
  eq5 <- y5 ~ y10, eq6 <- y6 ~ y5 + y9,
  eq7 <- y7 ~ y6, eq8 <- y8 ~ y12,
  eq9 <- y9 ~ y7, eq10 <- y10 ~ y5,
  eq11 <- y11 ~ y12, eq12 <- y12 ~ y4 + y11,
  eq13 <- y13 ~ y2 + y6
)

# ------------------------ #
#  Case A: diagonal Sigma  #
#  Figure 2                #
# ------------------------ #

Sigma = diag(length(eq.system))

#function call
decompose.A = causal_decompose(
  eq.system = eq.system,
  sigma.in  = Sigma
)

decompose.A$C
decompose.A$Psi0

#path diagram
tkplot(decompose.A$dec.graph)


# -------------------- #
#  Case B: free Sigma  #
#  Figure 3            #
# -------------------- #

# indexes of non-null elements of Sigma
sigma.idx = cbind(
  rbind(rep(1,5),c(4,5,8,10,12)), #y1
  rbind(rep(2,4),c(4,6,8,9)), #y2
  rbind(rep(3,4),c(6,7,11,13)), #y3
  rbind(rep(4,6),c(5,6,8,9,10,12)), #y4
  rbind(rep(5,3),c(8,10,12)), #y5
  rbind(rep(6,5),c(7,8,9,11,13)), #y6
  rbind(rep(7,2),c(11,13)), #y7
  rbind(rep(8,3),c(9,10,12)), #y8
  rbind(rep(10,1),c(12)), #y10
  rbind(rep(11,1),c(13)) #y11
  )

#fictitious Sigma matrix
low.tri <- as.matrix(Matrix::sparseMatrix(i = sigma.idx[2,] , j = sigma.idx[1,], x = 1,
                                           dims = rep(length(eq.system),2)))
Sigma = low.tri + t(low.tri) + diag(length(eq.system))

#function call
decompose.B=causal_decompose(
  eq.system = eq.system,
  sigma.in = Sigma)

decompose.B$C
decompose.B$Psi1
decompose.B$Psi0

#path diagram
tkplot(decompose.B$dec.graph)

# ------------------------------------- #
#  Dataset macroeconomic example        #
# ------------------------------------- #

#Load the data
data(macroIT)

# ----------- #
#  Example 1  #
# ----------- #

eq.system <- list(eq1 <- C ~  CP  + I +  CP_1  ,
                eq2 <- I ~ K + CP_1,
                eq3 <- WP ~  I + GDP +  GDP_1,
                eq4 <- GDP ~ C +I+ GDP_1,
                eq5 <- CP ~   WP  + T,
                eq6 <- K  ~ I + K_1)

instruments <- ~ T +  CP_1 + GDP_1 + K_1

#function call
dec.macroIT <- causal_decompose(data=macroIT,
                                eq.system=eq.system,
                                resid.est="noDfCor",
                                instruments = instruments)

summary(dec.macroIT$systemfit)

fitted.list=list()
for (j in 1:length(eq.system)){
  fitted.list[[j]]=dec.macroIT$systemfit$eq[[j]]$fitted.values
}
Y.hat=do.call(cbind,fitted.list)
Y.obs=macroIT[,unlist(lapply(lapply(eq.system, function(x) {
  all.vars(x)
}),
function(x) {
  x[1]
}))]

theil=list()
for (j in 1:length(eq.system)){
  y.hat=Y.hat[,j]
  y.obs=data.frame(Y.obs[,j])
  y.res=y.obs-y.hat
  colnames(y.obs)<-"y.obs"
  colnames(y.res)<-"y.res"
  Y.reg=cbind(y.hat,y.obs,y.res)
  theil[[j]]=sqrt(sum(y.res^2))/(sqrt(sum(y.obs^2))+sqrt(sum(y.hat^2)))
}
unlist(theil)

# ------------------------------------- #
#  Leave out non-significant estimates  #
# ------------------------------------- #

eq.system <- list(eq1 <- C ~ CP + CP_1  ,
                  eq2 <- I ~ K,
                  eq3 <- WP ~  I +  GDP_1,
                  eq4 <- GDP ~ C + I + GDP_1,
                  eq5 <- CP ~   WP  + T,
                  eq6 <- K  ~ I + K_1)

instruments <- ~ T +  CP_1 + GDP_1 + K_1

#function call
dec.macroIT.new <- causal_decompose(data = macroIT,
                                    eq.system = eq.system,
                                    resid.est = "noDfCor",
                                    instruments = instruments)

summary(dec.macroIT.new$systemfit)

dec.macroIT.new$C
dec.macroIT.new$Psi1
dec.macroIT.new$Psi0

fitted.list=list()
for (j in 1:length(eq.system)){
  fitted.list[[j]]=dec.macroIT.new$systemfit$eq[[j]]$fitted.values
}

Y.hat=do.call(cbind,fitted.list)
Y.obs=macroIT[,unlist(lapply(lapply(eq.system, function(x) {
  all.vars(x)
}),
function(x) {
  x[1]
}))]

theil.def=list()
for (j in 1:length(eq.system)){
  y.hat=Y.hat[,j]
  y.obs=data.frame(Y.obs[,j])
  y.res=y.obs-y.hat
  colnames(y.obs)<-"y.obs"
  colnames(y.res)<-"y.res"
  Y.reg=cbind(y.hat,y.obs,y.res)
  theil.def[[j]]=sqrt(sum(y.res^2))/(sqrt(sum(y.obs^2))+sqrt(sum(y.hat^2)))
}
unlist(theil.def)

#path diagram
tkplot(dec.macroIT.new$dec.graph)

# --------- #
#  Testing  #
# --------- #

test.E1=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 1,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E1$rho.est
test.E1$wald

test.E2=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 2,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E2$rho.est
test.E2$wald

test.E3=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 3,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E3$rho.est
test.E3$wald

test.E4=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 4,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E4$rho.est
test.E4$wald

#Singular W's
info.matrix <- solve(test.E4$rho.jacobian%*%solve(test.E4$theta.hessian)%*%t(test.E4$rho.jacobian))
test.E4$rho.est[1,2]^2*info.matrix[1,1]
test.E4$rho.est[2,2]^2*info.matrix[2,2]

test.E5=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 5,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E5$rho.est
test.E5$wald

test.E6=feedback_ml(data = macroIT, out.decompose = dec.macroIT.new,
                    eq.id = 6,
                    lb = min(dec.macroIT.new$Sigma)-10,
                    ub = max(dec.macroIT.new$Sigma)+10,
                    nrestarts = 10, nsim = 20000, seed.in = 1)
test.E1$rho.est
test.E1$wald

#Final graph
dec.edges <- get.edgelist(dec.macroIT.new$dec.graph)
id.delete <- c(which(dec.edges[,1] == "WP" & dec.edges[,2] == "CP"),
               which(dec.edges[,1] == "I"  & dec.edges[,2] == "GDP"))

final.graph <- delete_edges(dec.macroIT.new$dec.graph,
                            intersect(which(E(dec.macroIT.new$dec.graph)$lty==2),id.delete))

tkplot(final.graph)
