############################# The R Journal Submission ##############################
#####################################################################################
install.packages("mudfold") # install the package
library(mudfold) # load the package
set.seed(1) # set the seed for reproducibility of the results


#####################################################################################
################################ SECTION 2 ##########################################
#####################################################################################

data("ANDRICH") # load the data
?ANDRICH # check data description
N <- ncol(ANDRICH) # number of items
n <- nrow(ANDRICH) # number of persons
item_names <- colnames(ANDRICH) # item names



####################################################################
### Calculate Expected, Observed, and H coefficients for triples ###
####################################################################
experr <- mudfold:::Err_exp(ANDRICH) # Expected errors for triples
obserr <- mudfold:::Err_obs(ANDRICH) # Observed errors for triples
hcoeft <- 1 - (obserr / experr) # H coefficients for triples

## Symmetry in the H coefficients
triple_HDODE <- matrix(c("HIDEOUS","DONTBELIEV","DETERRENT"), ncol=3)
triple_DEDOH <- matrix(rev(triple_HDODE), ncol=3)
## Check of scalability equality in two symmetric triples
hcoeft[triple_HDODE] == hcoeft[triple_DEDOH]
## Or check equality among all symmetric triples
all.equal(hcoeft , aperm(hcoeft, c(3,2,1)))


####################################################################
##### FIRST STEP IN ITEM SELECTION:Find the best unique triple #####
####################################################################
library(gtools)

## Find unique (h,l,k) combinations of length three from N items.
perm1 <- combinations(N, 3, item_names, set = FALSE)
## Permute once to obtain the (h,k,l) combinations.
perm2 <- perm1[,c(1,3,2)]
## Permute again to obtain the (l,h,k) combinations.
perm3 <- perm1[,c(2,1,3)]

## Search for unique triples.
unq <- rbind(perm1[(hcoeft[perm1] > 0 & hcoeft[perm2] < 0 & hcoeft[perm3] < 0),],
                 perm2[(hcoeft[perm1] < 0 & hcoeft[perm2] > 0 & hcoeft[perm3] < 0),],
                 perm3[(hcoeft[perm1] < 0 & hcoeft[perm2] < 0 & hcoeft[perm3] > 0),])

## H coefficients for the set of unique triples 
hcoeft[unq]

## Specify lower bound for the scalability criterion
lambda1 <- 0.3

## Check if the criterion is fullfilled 
max(hcoeft[unq]) > lambda1 # if TRUE proceed to the second step


#####################################################################
#### SECOND STEP IN ITEM SELECTION:Extend the best unique triple ####
#####################################################################

## Find best unique triple. (== Starting scale for 2nd step)
BestUnique <- unq[which.max(hcoeft[unq]),] # Best unique triple

## Remaining items that are investigated in the 2nd step
ALLitems <- colnames(ANDRICH)
Remaining <- ALLitems[!ALLitems %in% BestUnique] # Remaining items
## Create indices to be used in constructing scales
lb <- length(BestUnique) # length of best unique triple
lr <- length(Remaining) # number of remaining items to add in the scale

## create all possible positions where each new item from Remaining
## can be added in the scale
index_rep <- rep(seq(1,(lb+1)),lr) -1  # vector with posit
index_irep <- rep(Remaining, each=lb+1)

## Create all possible scales by adding each item in Remaining
## to every possible position of BestUnique
ALLscales <- lapply(1:length(index_rep), 
                     function(i) append(BestUnique,index_irep[i],after = index_rep[i]))

## Example from p.6 of the paper that demonstrates 
## the first criterion of the second step in MUDFOLD's 
## item selection algorithm

Examplescale <- ALLscales[[1]]
Examplescale
les <- length(Examplescale)
ExamplescaleTRILES <- combinations(n=les, r=3, v=Examplescale, set=FALSE)
hcoeft[ExamplescaleTRILES[1:3,]]

#####################################################################################
################################ SECTION 3 ##########################################
#####################################################################################

## mudfold() function
fitANDRICH <- mudfold(ANDRICH, nboot=100, parallel="multicore", seed=1)
## generic functions
print(fitANDRICH) # print()
summary(fitANDRICH, boot=TRUE) # summary()
plot(fitANDRICH, plot.type="IRF") # plot()
plot(fitANDRICH, plot.type="persons") # plot()
plot(fitANDRICH, plot.type="scale") # plot()
coef(fitANDRICH, type = "items") # coef()
coef(fitANDRICH, type = "persons") # coef()
## diagnostics() function
diagnostics(fitANDRICH)
diagnostics(fitANDRICH, which = "H") # unidimensionality
diagnostics(fitANDRICH, which = "LI") # local independence 
diagnostics(fitANDRICH, which = "UM") # IRF unimodality
diagnostics(fitANDRICH, which = "MAX") # stochastic ordering == moving maxima 
diagnostics(fitANDRICH, which = "STAR") # moving maxima 
diagnostics(fitANDRICH, which = "ISO") # manifest unimodality



#####################################################################################
################################ SECTION 4 ##########################################
#####################################################################################

#####################################################################
############# MUDFOLD ANALYSIS OF THE LONELINESS SCALE ##############
#####################################################################

data("Loneliness")
dat <- pick(Loneliness, cutoff = 3) # dichotomize responses
Lonelifit <- mudfold(dat, lambda1=0 ,nboot = 100, seed = 1) # fit mudfold scale
loneliSummary <- summary(Lonelifit, boot=TRUE)
loneliScale <- loneliSummary$ITEM_STATS$ITEM_DESCRIPTIVES$items
loneliScale
loneliSummary$SCALE_STATS[1:3,]

## other diagnostics
par(mfrow=c(1,2))
# testing for local independence
diagnostics(Lonelifit, which = "LI") 
# visual inspection of moving maxima 
diagnostics(Lonelifit, which = "STAR")
par(mfrow=c(2,4))
# visual inspection for IRF unimodality
diagnostics(Lonelifit, which = "UM")
par(mfrow=c(1,1))

loneliSummary$ITEM_STATS$H_MUDFOLD_items
loneliSummary$ITEM_STATS$ISO_MUDFOLD_items
loneliSummary$ITEM_STATS$MAX_MUDFOLD_items

plot(Lonelifit,plot.type = "IRF")
plot(Lonelifit,plot.type = "persons")

#####################################################################
############# MUDFOLD ANALYSIS OF THE PLATOS 7 WORKS ################
#####################################################################

data("Plato7")
dat.Plato <- pick(Plato7)
fitPlato <- mudfold(dat.Plato, nboot = 100, seed = 1)
summaryPlato <- summary(fitPlato, boot=TRUE)
summaryPlato$SCALE_STATS[1:3,]

summaryPlato$ITEM_STATS$H_MUDFOLD_items
summaryPlato$ITEM_STATS$ISO_MUDFOLD_items

plot(fitPlato, plot.type="IRF")
par(mfrow=c(2,3))
diagnostics(fitPlato, which = "UM")
par(mfrow=c(1,1))


## Additional steps that can be taken
## and other diagnostics
par(mfrow=c(1,2))
# testing for local independence
diagnostics(fitPlato, which = "LI") 
# visual inspection of moving maxima 
diagnostics(fitPlato, which = "STAR")
par(mfrow=c(1,1))
summaryPlato$ITEM_STATS$MAX_MUDFOLD_items

## new search using as a starting scale the scale from the first 
## analysis and relaxing values for lambda2
start.new <- summaryPlato$ITEM_STATS$ITEM_DESCRIPTIVES$items
fitPlato2 <- mudfold(dat.Plato,lambda2 = -0.5, start.scale = start.new, nboot = 100, seed = 1)
## get the descriptive summary of the scale
summary(fitPlato2, boot=TRUE)$ITEM_STATS$ITEM_DESCRIPTIVES ## Timaeus has been added second in the scale
## check the empirical item response functions
plot(fitPlato2, plot.type = "IRF")
## check the item response functions
par(mfrow=c(2,3))
diagnostics(fitPlato2, which = "UM")
par(mfrow=c(1,1))
