library(rcosmo)
citation("rcosmo")
library(gsl)
library(celestial)
library(microbenchmark)

#====================Getting data into rcosmo
filename <- "CMB_map_smica1024.fits"
downloadCMBMap(foreground = "smica", nside = 1024, filename)
sky<-CMBDataFrame(filename)
str(sky)

set.seed(0); s <- 2e6;
cmb_sample <- CMBDataFrame(filename, sample.size = s, include.m = T, include.p = T)
cmb_sample

#====================Use of memory mapping

filename1 <- "CMB_map_smica2048.fits"
downloadCMBMap(foreground = "smica", nside = 2048, filename1)
system.time(sky <- CMBDataFrame(filename1))

system.time(fits <- FITSio::readFITS(filename1))

v <- c(1,2,4,7,11)
sky <- CMBDataFrame(filename, spix = v, include.p = T, include.m = T, coords = "spherical")
sky
pix(sky)

#====================Introduction to Healpix

ns <- 64; rand <- rnorm(12 * ns ^ 2)
cmbdf <- CMBDataFrame(nside = ns, I = rand, ordering = "nested")
w1 <- window(cmbdf, in.pixels = c(1,9))
w2 <- window(cmbdf, in.pixels = c(2,10))
w3 <- window(cmbdf, in.pixels = c(4,12))
plot(w1, col = "blue", back.col = "white", xlab = '', ylab = '', 
     zlab = '')
plot(w2, col = "green", add = TRUE)
plot(w3, col = "orange", add = TRUE)
displayPixelBoundaries(nside = 1, ordering = "nested", incl.labels = 1:12, col =
"red")

ns <- 256; rand <- rnorm(12 * ns ^ 2)
w21 <- window(CMBDataFrame(nside = ns, I = rand, ordering = "nested"), in.pixels = 1)
plot(w21, col = "light blue", back.col = "white", add = TRUE, size = 1.2)
displayPixelBoundaries(nside = 2, ordering = "nested", incl.labels = c(1,2,3,4), col
="black")
plot(window(cmbdf,in.pixels = 2), col = "green", add = TRUE)
plot(window(cmbdf,in.pixels = 4), col = "purple", add = TRUE)
plot(window(cmbdf,in.pixels = 5), col = "orange", add = TRUE)
plot(window(cmbdf,in.pixels = 6), col = "red", add = TRUE)
 
#----
cmbdf <- CMBDataFrame(nside = 8, ordering = "ring")
plot(cmbdf, type = 'l', col = "black", back.col = "white")
tolabel <- c(1,100:107,768)
plot(cmbdf[tolabel,], labels = tolabel, col = "red", add = TRUE)

#====================HEALPix functions

sky <- CMBDataFrame(nside = 2, ordering = "ring"); ordering(sky)
ordering(sky) <- "nested"; ordering(sky)

ancestor(1e3, 1:5)

p <- 1; j1 <- 1; j2 <- 5
P <- pixelWindow(j1 = j1, j2 = j2, pix.j1 =  p)
displayPixels(spix = P, j = j2, plot.j = j2)

geoDistSearch <- function(target, nside, spix) {
  xyz <- pix2coords(nside = nside, spix = spix)
  dists <- geoDist(xyz, target)
  return(xyz[which.min(dists),])
}

t <- data.frame(x = 0.6, y = 0.8, z = 0)
nside <- 16; p <- 1:(12*(nside)^2)
mb <- microbenchmark::microbenchmark(
  geoDistSearch(target = t, nside = nside, spix = p),
  nestSearch(target = t, nside = nside))
summary(mb)$median[1]/summary(mb)$median[2]

#========Subsetting and combining spherical regions

download.file("https://www.iau.org/static/public/constellations/txt/dra.txt",
              "bound1.txt")
x1 <- readLines("bound1.txt")
x1 <- gsub("\\|", " ", x1)
Constellation_Boundary1 <- read.table(text = x1,col.names=c("H","M","S","D","Con_N"))
download.file("https://www.iau.org/static/public/constellations/txt/sgr.txt",
              "bound2.txt")
x2 <- readLines("bound2.txt")
x2 <- gsub("\\|", " ", x2)
Constellation_Boundary2 <- read.table(text = x2,col.names=c("H","M","S","D","Con_N"))

CB0 <- Constellation_Boundary1
deg <- celestial::hms2deg(CB0[,1],CB0[,2],CB0[,3])
CB1 <- data.frame(pi*deg/180, pi*CB0[,4]/90)
colnames(CB1) <- c("phi","theta")
polygon1 <- CMBWindow(phi = CB1$phi, theta = CB1$theta)
plot(cmb_sample, back.col = "white")
plot(polygon1, lwd=2)

CB0 <- Constellation_Boundary2
deg <- celestial::hms2deg(CB0[,1],CB0[,2],CB0[,3])
CB1 <- data.frame(pi*deg/180, pi*CB0[,4]/90)
colnames(CB1) <- c("phi","theta")
polygon2 <- CMBWindow(phi = CB1$phi, theta = CB1$theta)
plot(cmb_sample, back.col = "white")
plot(polygon2, lwd=2)


# Exterior of a disc of radius 0.5
d.exterior <- CMBWindow(theta = pi/2, phi = 0, r = 0.5,
set.minus = TRUE)
# List containing d.exterior and the interior of a disc of r = 1
wins <- list(d.exterior, CMBWindow(theta = pi/2, phi = 0, r = 1))
# sky.annulus

df <- CMBDataFrame(filename)
sky.annulus <- window(df, new.window = wins)
plot(sky.annulus, back.col = "white", xlab = '', ylab = '', 
     zlab = '')
plot(wins[[1]], lwd = 5, col="blue"); plot(wins[[2]], lwd = 5, col="blue")
#==============Spherical geometry functions

geoArea(sky.annulus)

p <- c(0,0,1)
maxDist(sky.annulus, p)


win1 <- triangulate(polygon2)
for (i in 1:11){
plot(win1[[i]],  col=i)
}


#=============Univariate  Spherical Statistics and Plots
set.seed(0)
sampleCMB(df, sample.size = 3)


summary(sky.annulus)

entropyCMB(cmbdf = df, win = d.exterior, intensities = "I")

fmf(cmbdf = sky.annulus, alpha = 0, intensities = "I")

Tq <- fRen(cmbdf = sky.annulus, q.min = 1.01, q.max = 10, N = 20,
          intensities = "I")
plot(Tq[,1], Tq[,2], ylab =expression(T[q]), xlab = "q",
main = "Sample Renyi function", pch = 20, col = "blue")

df1 <- sampleCMB(df, sample.size = 100000)
cmbdf.win <- window(df1, new.window = polygon2)
plotAngDis(cmbdf.win, intensities = "I")

#====== Multivariate Statistics for Data From Different CMBWindows

#qqplotWin and chi2CMB may be rather lengthy

qqplotWin(df1, polygon1, polygon2)

lw <- list(polygon1, polygon2)
qstat(df1, lw)


#=========== Investigating Spatial Dependencies
covmodelCMB(pi/4, cov.pars = c(1, pi), kappa = 4, cov.model = "askey" )

plotcovmodelCMB("askey", phi = pi/4, to  = pi/2, kappa = 4)

df1 <- sampleCMB(df, sample.size = 100000)
Cov <- covCMB(df1, max.dist = 0.03, num.bins = 10)
Cov$v
plot(Cov)

varcmb <- variogramCMB(df1, max.dist = 0.1, num.bins = 30)
ols <- variofitCMB(varcmb,  fix.nug=FALSE, wei="equal", cov.model= "matern")
plot(varcmb)
lines(ols, lty=2)

COM_PowerSpectra <- downloadCMBPS(link=1)
Cov_est <- covPwSp(COM_PowerSpectra[,1:2], 2000)
# Plot the covariance estimate as a function of angular distances
plot(acos(Cov_est[,1]), Cov_est[,2], type ="l",
     xlab ="angular distance", ylab ="Estimated Covariance")


#=========== Converting other spherical data to HEALPix format 

# First download and unpack data from https://simplemaps.com/data/world-cities

worldcities <- read.csv("worldcities.csv")

## Prepare a data frame with cities' coordinates
sph <- geo2sph(data.frame(lon = pi/180*worldcities$lng,
                          lat = pi/180*worldcities$lat))
df2 <- data.frame(phi = sph$phi,
                 theta = sph$theta,
                 I = rep(1,nrow(sph)))

## Create and plot the corresponding HPDataFrame 
## by choosing a resolution (nside = 102)
hp <- HPDataFrame(df2, auto.spix = TRUE, nside = 1024)
plot(hp, size = 3, col = "darkgreen", back.col = "white", xlab = '', ylab = '',      zlab = '')




