# Section - Contents of EpiILM
# Subsection - Simulation of epidemics
# Subsubsection - Example: spatial model

# install EpiILM package from CRAN
install.packages("EpiILM")
# Load the EpiILM package
library("EpiILM")
# simulate (x,y) coordinates
set.seed(101)
x <- runif(100, 0, 10)
y <- runif(100, 0, 10)
#  simulate epidemic 
SI.dis <- epidata(type = "SI", n = 100, tmax = 15, sus.par = 0.3, beta = 5.0,
               x = x, y = y)
SI.dis$inftime

# epidemic curve for SI.dis - figure 1
plot(SI.dis, plottype = "curve", curvetype ="complete")

# summary graphics for the simulated epidemic - figure 2
plot(SI.dis, plottype = "spatial")

#######################################################################################
# Section - Contents of EpiILM
# Subsection - Simulation of epidemics
# Subsubsection - Example: network model

#  generate contact network matrix
set.seed(101)
contact <- matrix(rbinom(10000, 1, 0.1), nrow = 100, ncol = 100)
diag(contact[,])  <- 0
# plot contact network - figure 3
require("igraph")
net1 <- graph_from_adjacency_matrix(contact)
plot(net1, vertex.size = 10, vertex.label.cex = 0.5, edge.arrow.mode = "-")
# generate covariate and simulate epidemic 
set.seed(101)
Z <- round(runif(100, 0, 2))
SI.contact <- epidata(type = "SI", Sformula = ~Z, n = 100, tmax = 15,
                sus.par = c(0.1, 0.05),  contact = contact)
SI.contact$inftime
#  epidemic curve for SI.contact - figure 4
plot(SI.contact, plottype = "curve", curvetype ="complete")

#######################################################################################
# Subsection - Bayesian Inference
# Subsubsection - Example: spatial model

#  generate covariate and simulate epidemic with Fortran seed
set.seed(101)
# number of animals on each farm
A <- round(rexp(100, 1/50))
SI.dis.cov <- epidata(type = "SI", n = 100, tmax = 50, x = x, y = y,
                 Sformula = ~A, sus.par = c(0.2, 0.1), beta = 5)

# The last time point at which infection is observed
t_end <- max(SI.dis.cov$inftime)

# Uniform prior for alpha parameters
unif_range <- matrix(c(0, 0, 10000, 10000), nrow = 2, ncol = 2)

# MCMC function to estimate Model 7 parameters
set.seed(1001)
mcmcout_Model7 <- epimcmc(SI.dis.cov, Sformula = ~A, tmax = t_end, niter = 50000, 
                      sus.par.ini = c(0.001, 0.001), beta.ini = 0.01, 
                      pro.sus.var = c(0.01, 0.01), pro.beta.var = 0.5,
                      prior.sus.dist = c("uniform","uniform"), 
                      prior.sus.par = unif_range, 
                      prior.beta.dist = "uniform", prior.beta.par = c(0, 10000))
summary(mcmcout_Model7, start = 10001)

# Trace plot after  10000  burn-in - figure 5
plot(mcmcout_Model7, partype = "parameter", start = 10001, density = FALSE)

#######################################################################################
# Subsection - Bayesian Inference
# Subsubsection - Example: network model

set.seed(101)
# sample size
n <- 500

# generate contact network
contact <- matrix(rbinom(n*n, size = 1, prob = 0.1), nrow = n, ncol = n)
diag(contact) <- 0

# generate number of sheep and cows 
X1 <- round(rexp(n, 1/100))
X2 <- round(rgamma(n, 50, 0.5))

# simulate epidemic form SIR network-based ILM
infp <- rep(3, n)
SIR.net <- epidata(type = "SIR", n = 500, tmax = 25, 
                                     sus.par = c(0.003, 0.01), trans.par = c(0.0003, 0.0002), 
                                     contact = contact, infperiod = infp,
                                     Sformula = ~ -1 + X1 + X2, Tformula = ~ -1 + X1 + X2)

#  epidemic curve for SI.contact - figure 6
plot(SIR.net, plottype = "curve", curvetype = "complete")

# MCMC function to estimate Model 8 parameters
t_end <- max(SIR.net$inftime)
prior_par <- matrix(rep(1, 4), ncol = 2, nrow = 2)
mcmcout_SIR.net <- epimcmc(SIR.net, tmax = t_end, niter = 50000,
                                      Sformula = ~-1 + X1 + X2, Tformula = ~-1 + X1 + X2,  
                                      sus.par.ini = c(0.003, 0.001), trans.par.ini = c(0.01, 0.01),
                                      pro.sus.var = c(0.0, 0.1), pro.trans.var = c(0.05, 0.05),
                                      prior.sus.dist = c("gamma", "gamma"), prior.trans.dist = c("gamma", "gamma"), 
                                      prior.sus.par = prior_par, prior.trans.par = prior_par,
                                      adapt = TRUE, acc.rate = 0.5)
                                      
# Trace plot after  10000  burn-in - figure 7
plot(mcmcout_SIR.net, partype = "parameter", start = 10001, density = FALSE)

# summary of MCMC results
summary(mcmcout_SIR.net, start = 10001) 

#######################################################################################
# Subsection: Case study Tomato spotted wilt virus (TSWV) data
# Load the data
data(tswv)
# Define the data for the analysis
x <- tswv$x
y <- tswv$y
inftime     <- tswv$inftime
removaltime <- tswv$removaltime
infperiod   <- rep(3, length(x))
epidat.tswv <- as.epidata(type = "SIR", n = 520, x = x, y = y, inftime = inftime, infperiod = infperiod)
                               
# Spatial plot - figure 8
plot(epidat.tswv, plottype = "spatial", tmin = 2)

set.seed(101)
# MCMC
mcmc.tswv <- epimcmc(epidat.tswv, tmin = 2, tmax = 10,
                niter = 50000, sus.par.ini = 0.0, beta.ini = 0.01,
                pro.sus.var = 0.000005, pro.beta.var = 0.005, prior.sus.dist = "gamma",
                prior.sus.par = c(1,10**(-3)), prior.beta.dist = "gamma", prior.beta.par = c(1,10**(-3)))

# MCMC results summary
summary(mcmc.tswv, start = 10001)

# trace plot - figure 9
plot(mcmc.tswv, partype = "parameter", start = 10001, density = FALSE)

#######################################################################################
