## ----setup_paper, include=FALSE-----------------------------------------------
if (F){ ## From reproducing, please use this dev ver:
  install_version("ggplot2", version = "0.2.5", repos = "http://cran.us.r-project.org")
}

knitr::opts_chunk$set(
  fig.align  = "center",
  echo       = TRUE,
  collapse   = TRUE,
  message    = FALSE,
  warning    = FALSE,
  error      = FALSE,
  cache      = FALSE,
  cache.lazy = FALSE
)


library(tourr)
library(spinifex) ## Use dev version listed above!
library(ggplot2)
library(gridExtra)
library(dplyr)

my_theme <- list(scale_color_brewer(palette = "Dark2"),
                 theme_void(), 
                 theme(legend.position ="none"))


## ----paperFunctions, include=FALSE--------------------------------------------
## FUNCTION FOR STATIC OUTPPUT, 
# going to facets loses control of multiple geoms and output size, simpler to just live with this.
array2ggfacets <- function(tour_array, data, m_var, class, margin = 2.2){
  n_frames <- dim(tour_array)[3]
  if (n_frames != 15) 
    stop(paste0("stop: n_frames != 15!!! Check the angle step size. n_frames = ", n_frames))
  
  ## Initialize
  frames       <- array2df(array = tour_array, data = data)
  basis_frames <- frames$basis_slides
  data_frames  <- frames$data_slides
  p            <- nrow(basis_frames) / n_frames
  
  ## manip var asethetics
  col_v        <- rep("grey80", p)
  col_v[m_var] <- "blue"
  col_v        <- rep(col_v, n_frames)
  siz_v        <- rep(0.3, p)
  siz_v[m_var] <- 1
  siz_v        <- rep(siz_v, n_frames)
  cat          <- rep(as.factor(class), n_frames)

  ## circle
  angle <- seq(0, 2 * pi, length = 180)
  circ  <- data.frame(c_x = cos(angle), c_y = sin(angle))
  circ[nrow(circ)+1, ] <- NA
  ## Data asethetics
  data_frames <- data.frame(data_frames, class = rep(class, n_frames))
  colnames(data_frames)  <- c("x", "y", "frame", "class")
  colnames(basis_frames) <- c("x", "y", "frame", "lab")
  grid_b <- grid_t <- data.frame(
    frame = 1:n_frames, x = margin * rep(1:5, 3), y = margin * rep(3:1, each = 5))
  grid_t$y <- grid_t$y + max(grid_t$y)
  ## OUTER JOIN
  basis_grid <- merge(x = basis_frames, y = grid_t, by = "frame", all = TRUE)
  ## CROSS JOIN
  circ_grid  <- merge(x = circ, y = grid_t, by = NULL)
  ## OUTER JOIN
  data_grid  <- merge(x = data_frames, y = grid_b, by = "frame", all = TRUE)
  
  ##### RENDER
  ## SETUP
  gg <- ggplot(data = basis_grid) +
    ## AXES LINE SEGMETNS
    geom_segment(aes(x = x.x + x.y, y = y.x + y.y, xend = x.y, yend = y.y),
                 color = col_v, size = siz_v) +
    ## AXES TEXT LABELS
    geom_text(aes(x = x.x + x.y, y = y.x + y.y, label = lab),
              color = col_v, vjust = "outward", hjust = "outward") +
    ## AXES FRAME NUM
    geom_text(aes(x = x.y - .7, y = y.y + 1.1, 
                  label = paste0("frame: ", frame)), color = "grey50") +
    ## AXES CIRCLE PATH
    suppressWarnings( # Suppress for "Removed 1 rows containing missing values."
      geom_path(data = circ_grid, color = "grey50",
                mapping = aes(x = x + c_x, y = y + c_y))
    )
  
  ## PROJECTION
  gg <- gg +
    ## PROJ DATA POINTS
    geom_point(data = data_grid, size = .7,
               mapping = aes(x = x.x + x.y, y = y.x + y.y, 
                             color = class, shape = class)) +
    ## FACET FRAME NUM
    geom_text(data = data_grid, color = "grey50",
              mapping = aes(x = x.y - .7, y = y.y + 1.1, 
                            label = paste0("frame: ",frame))) +
    my_theme
  
  ## Return
  gg
}



view_manip_space2 <- function (basis,
                               manip_var,
                               manip_col = "blue",
                               tilt = pi * 5 / 12,
                               z_col = "red",
                               lab = paste0("V", 1:nrow(basis)))
{
  #### Finds the angle between two vectors
  find_angle <- function(a, b)
    acos(sum(a * b) / (sqrt(sum(a * a)) * sqrt(sum(b * b))) )
  #### Makes a df semi-circle, with 1 row per degree
  make_curve <- function(ang_st = 0,
                         ang_stop = 2 * pi) {
      degrees <- round(360 / (2 * pi) * abs(ang_st - ang_stop))
      angle <- seq(ang_st, ang_stop, length = degrees)
      data.frame(x = cos(angle), y = sin(angle), z = sin(angle))
    }
  
  ## Initialize
  p <- nrow(basis)
  m_sp   <- as.data.frame(create_manip_space(basis, manip_var))
  colnames(m_sp) <- c("x", "y", "z")
  m_sp_r <- m_sp %>% mutate(y = y * cos(tilt), z = z * sin(tilt))
  mvar   <- m_sp[manip_var, ]
  mvar_r <- m_sp_r[manip_var, ]
  
  col_v <- rep("grey80", p)
  col_v[manip_var] <- manip_col
  siz_v <- rep(0.3, p)
  siz_v[manip_var] <- 1
  
  circ_r <- make_curve() %>%
    mutate(y = y * cos(tilt), z = z * sin(tilt))
  
  phi_ang     <- find_angle(c(m_sp_r$x, m_sp_r$y), c(m_sp_r$x, m_sp_r$z))
  phi_offset  <- find_angle(c(mvar_r$x, mvar_r$y), c(1,0))
  phi_curve_r <- .4 * make_curve(ang_st = phi_offset, ang_stop = phi_ang) %>%
    mutate(y = y * cos(tilt), z = z * sin(tilt))
  
  thata_ang     <- find_angle(c(mvar$x, mvar$y),c(1, 0))
  theta_curve_r <- .5 * make_curve(ang_st = 0, ang_stop = thata_ang) %>%
    mutate( y=y*cos(tilt), z=z*sin(tilt))
  
  ## Render
  gg <-
    ggplot2::ggplot() + my_theme +
    ## Circle
    ggplot2::geom_path(data = circ_r,
                       mapping = ggplot2::aes(x = x, y = y),
                       color = manip_col, size = 0.3, inherit.aes = FALSE) +
    ## Projection plane:
    ggplot2::geom_segment(data = m_sp_r,
                          mapping = ggplot2::aes(
                            x = x,
                            y = y,
                            xend = 0,
                            yend = 0
                          ),
                          size = siz_v, colour = col_v) +
    ggplot2::geom_text(data = m_sp_r,
                       mapping = ggplot2::aes(x = x, y = y, label = lab),
                       size = 4, colour = col_v,
                       vjust = "outward", hjust = "outward") +
    ## Z direction
    ggplot2::geom_path(data = circ_r,
                       mapping = ggplot2::aes(x = x, y = z),
                       color = z_col, size = 0.3, inherit.aes = FALSE) +
    ggplot2::geom_segment(data = mvar_r,
                          mapping = ggplot2::aes(x = x,y = z,
                                                 xend = 0,yend = 0),
                          size = 1, colour = z_col) +
    ggplot2::geom_segment(data = mvar_r,
                          mapping = ggplot2::aes(x = x, y = z,
                                                 xend = x,yend = y),
                          size = 0.3, colour = "grey80", linetype = 2) +
    ggplot2::geom_text(data = mvar_r,
                       mapping = ggplot2::aes(x = x, y = z, 
                                              label = lab[manip_var]),
                       size = 4, colour = z_col,
                       vjust = "outward",
                       hjust = "outward"
    ) +
    ## Label angles
    geom_path(data = phi_curve_r,
              mapping = aes(x = x, y = z),
              color = z_col, size = 0.2) +
    geom_text(data = 1.2 * phi_curve_r[ceiling(nrow(phi_curve_r) / 2), ],
              mapping = ggplot2::aes(x=x, y=z, label = "phi"),
              color = z_col, size = 4, parse = TRUE) +
    geom_path(data = theta_curve_r,
              mapping = aes(x, y),
              color = manip_col,
              size = 0.2) +
    geom_text(data = 1.2 * theta_curve_r[ceiling(nrow(theta_curve_r)/2), ],
              mapping = ggplot2::aes(x = x, y = y - .02, label = "theta"),
              color = manip_col, size = 4, parse = TRUE)
  
  ## Return
  gg
}


## ----step0, echo=F, results='hide', out.width='70%',                                 fig.cap="Initial 2D projection: representation of the basis (left) and resulting data projection (right) of standardized flea data. The color and shape of data points are mapped to beetle species. The basis was identified using a projection pursuit guided tour, with the holes index. The contribution of the variables aede2 and tars1 approximately contrasts the other variables. The visible structure in the projection are the three clusters corresponding to the three species. Produced with the function \\code{view\\_basis()}."----
## Flea holes tour
set.seed(20190425)
f_dat <- tourr::rescale(flea[,1:6])
f_clas <- factor(flea$species)
## hard code a basis. tourr doesn't fix results with set.seed().
f_bas <- c(.693, -.022, .082, -.119, .706, .023, 
           -.070, .438, .405, .515, .103, .604) %>% 
  matrix( ncol=2) %>% 
  tourr::orthonormalise()
f_mvar <- 5
f_msp <- create_manip_space(basis = f_bas, manip_var = f_mvar)
f_proj <- data.frame(tourr::rescale(f_dat %*% f_msp[, 1:2]))

# step0, output
# Adjust centering to make axes and data side by side
oblique_frame(f_bas, data = f_dat, manip_var = f_mvar, lab = colnames(f_dat),
              color = f_clas, shape = f_clas) + 
  my_theme


## ----step2, echo=F, out.width='70%',                                                fig.cap="Illustration of a 3D manip space, this space is rotated effectively changing the contribution of the manip variable, aede2 in the example data. The blue circle and variable map lies on the projection plane. The red circle, orthogonal to the projection plane, illustrates the manipulation space and how the manip var can be controlled and how this affects the variable contribution back onto the projection plane. The other variables are omitted from the manipulation dimension for simplicity. Picturing the other variables in that dimension reveals the intuition that rotating one variable performs a constrained rotation on the others. This is illustrated with the \\code{view\\_manip\\_space()} function."----
view_manip_space2(basis = f_bas, manip_var = f_mvar, lab = colnames(f_dat))


## ----step3, echo=F, warning=F, out.width='100%',                                     fig.cap="Snapshots of a radial manual tour manipulating aede2: (1) original projection, (2) full contribution, (3) zero contribution, (4) back to original. "----
f_angle <- .29
f_mtour <- manual_tour(f_bas, manip_var = f_mvar, angle = f_angle)
if(dim(f_mtour)[3] != 15) message(
  paste0("!!! n_frames != 15 !!! Check angle step size. n_frames = ", dim(f_mtour)[3], ". angle = ", f_angle))
ggproto
z <- data.frame(0,0,0,0,0,0)

p1 <- oblique_frame(f_mtour[,, 1], data = z, lab = colnames(f_dat), manip_var = f_mvar) +
  ggtitle("(1) phi = 0.78") + my_theme
p2 <- oblique_frame(f_mtour[,, 5], data = z, lab = colnames(f_dat), manip_var = f_mvar) +
  ggtitle("(2) phi = 0") + my_theme
p3 <- oblique_frame(f_mtour[,, 12], data = z, lab = colnames(f_dat), manip_var = f_mvar) +
  ggtitle("(3) phi = 1.57") + my_theme
p4 <- oblique_frame(f_mtour[,, 15], data = z, lab = colnames(f_dat), manip_var = f_mvar) +
  ggtitle("(4) phi = 0.78") + my_theme
grid.arrange(p1, p2, p3, p4, ncol = 4)


## ----step4, echo=F, warning=F, out.height='7in', out.width='5.83in',                 fig.cap="Radial manual tour manipulating aede2 of standardized flea data. The axis for aede2 increases in contribution to the projection, from its initial value to 1, decreasing to 0 and then returning to the initial value. This effects the separation between the purple and green clusters. This shows that aede2 is important for distinguishing the purple species, because the separation disappears when aede2 is not contributing to the projection. An animation can be viewed at https://nspyrison.netlify.com/thesis/flea\\_manualtour\\_mvar5/."----
## Figure
array2ggfacets(tour_array = f_mtour, data = f_dat, m_var = f_mvar, class = f_clas)

## Saving gif for the supplementary material:
if (FALSE){
  play_manual_tour(basis = f_bas, data = f_dat, manip_var = f_mvar,
                   color = f_clas, shape = f_clas,
                   ggtheme = list(theme_void(), theme(legend.position = "none")),
                   fps = 8, start_pause = .5, end_pause = 1,
                   render_type = render_gganimate,
                   gif_filename = "flea_radialtour_mvar5.gif", 
                   gif_path = "./gifs")
}


## ----installCODEEXAMPLE, eval=F, echo=T---------------------------------------
#> ## Install from CRAN
#> install.package("spinifex")
#> ## Load into session
#> library("spinifex")


## ----installCODEEXAMPLE2, eval=F, echo=T--------------------------------------
#> ## Shiny app for visualizing basic application
#> run_app("intro")
#> ## View the code vignette
#> vignette("spinifex_vignette")


## ----installCODEEXAMPLE3, eval=F, echo=T--------------------------------------
#> ## Optionally install latest developmention version from GitHub
#> remotes::install_github("nspyrison/spinifex")


## ----functionsTable, echo=F---------------------------------------------------
library(kableExtra)

funcs <- rbind(
  c("construction", "create_manip_space", "forms the 3D space of rotation"),
  c("construction", "rotate_manip_space", "performs 3D rotation"),
  c("construction", "manual_tour", "generates sequence of 2D frames"),
  c("", "", ""),
  c("render", "array2df", "turn the tour path array into long form, for plotting"),
  c("render", "render_", "render long form as a ggplot2 objection for animation"),
  c("render", "render_plotly", "render the animation as a plotly object (default)"),
  c("render", "render_gganimate", "render the animation as a gganimate object"),
  c("", "", ""),
  c("animation", "play_tour_path", "composite function animating the specified tour path"),
  c("animation", "play_manual_tour", "composite function animating the specified manual tour"),
  c("", "", ""),
  c("specialty", "print_manip_space", "table of the rotated basis and manip space"),
  c("specialty", "oblique_frame", "display the reference axes of a given basis"),
  c("specialty", "view_manip_space", "illustrative display of any manip space")
)
colnames(funcs) <- c("Type", "Function", "Description")
  
kable(funcs, "latex", caption = "Summary of available functions.", booktabs = T, linesep="") 


## ----step0CODEEXAMPLE, eval=F, echo=T-----------------------------------------
#> library(spinifex)
#> ## Standardized flea data
#> f_data <- tourr::rescale(flea[, 1:6])
#> ## Guided tour path, holes index
#> f_path <- save_history(f_data, guided_tour(holes()))
#> ## Local extrema found
#> f_basis <- matrix(f_path[,, max(dim(f_path)[3])], ncol=2)
#> ## Categorical class variable
#> f_clas <- factor(flea$species)
#> ## Manip var, number of the variable to alter
#> f_mvar <- 5
#> ## Anglular dist between frames (radians)
#> step_size <- .26
#> ## Render and play animate, as plotly object by default
#> play_manual_tour(data = f_data,
#>                  basis = f_basis,
#>                  manip_var = f_mvar,
#>                  angle = step_size,
#>                  col = f_clas,
#>                  pch = f_clas)


## ----eval=FALSE---------------------------------------------------------------
#> ## View a basis and projected data
#> oblique_frame(basis = f_basis,
#>               data = f_data,
#>               color = f_clas,
#>               shape = f_clas)


## ----step2CODEEXAMPLE, eval=F, echo=T-----------------------------------------
#> ## Displays the projection plane and manipulation space for the
#> view_manip_space(basis = f_basis,
#>                  manip_var = f_mvar,
#>                  lab = colnames(f_data))


## ----JetClusterGood, echo=F, warning=F, out.height='7in', out.width='5.83in',        fig.cap="Snapshots of a radial manual tour of PC4 within the jet cluster, with color indicating experiment type: ATLAS7new (green) and ATLAS7old (orange). When PC4 is removed from the projection (frame 10) there is little difference between the groups, suggesting that PC4 is important for distinguishing the experiments. The animation can be viewed at https://nspyrison.netlify.com/thesis/jetcluster\\_manualtour\\_pc4/."----
# Jet cluster, Fig 7 of cook_dynamical_2018, subset to ATLAS7old and ATLAS7new
load("./data/JetCluster_sub.rda")
load("./data/JetCluster_basis.rda")

jet_bas   <- JetCluster_basis
jet_dat   <- tourr::rescale(JetCluster_sub[, 1:4])
jet_clas  <- factor(JetCluster_sub$exp)
jet_mvar  <- 4
jet_ang   <- .315
jet_mtour <- manual_tour(basis = jet_bas, manip_var = jet_mvar, angle = jet_ang)

## Figure
array2ggfacets(tour_array = jet_mtour, data = jet_dat,
               m_var = jet_mvar, class = jet_clas)

## Saving gif of radial tour for each PC; for the supplementary material
if (FALSE){
  for(i in 1:ncol(jet_dat)){
    fn <- paste0("jetcluster_radialtour_pc", i, ".gif")
    play_manual_tour(basis = jet_bas, data = jet_dat, manip_var = i,
                     color = jet_clas, shape = jet_clas,
                     ggtheme = list(theme_void(), theme(legend.position = "none")),
                     fps = 8, start_pause = .5, end_pause = 1,
                     render_type = render_gganimate,
                     gif_filename = fn, 
                     gif_path = "./gifs")
  }
}

##M_VAR, RESULT
# 1, good
# 2, poor
# 3, *worst
# 4, *best


## ----JetClusterBad, echo=F, warning=F, out.height='7in', out.width='5.83in',         fig.cap="Snapshots of a radial manual tour of PC3 within the jet cluster, with color indicating experiment type: ATLAS7new (green) and ATLAS7old (orange).  When the contribution from PC3 is changed there is little change to the structure of the two groups, suggesting that PC3 is not important for distinguishing the experiments. The animation can be viewed at https://nspyrison.netlify.com/thesis/jetcluster\\_manualtour\\_pc3/."----
# Jet cluster, Fig 7 of cook_dynalical_2018, subset to ATLAS7old and ATLAS7new
jet_mvar  <- 3
jet_mtour <- manual_tour(basis = jet_bas, manip_var = jet_mvar, angle = jet_ang)

## Figure
array2ggfacets(tour_array = jet_mtour, data = jet_dat,
               m_var = jet_mvar, class = jet_clas)


## ----DISclusterGood, echo=F, warning=F, out.height='7in', out.width='5.83in',        fig.cap="Snapshots of a radial manual tour exploring the sensitivity PC6 has on the structure of the DIS cluster, with color indicating experiment type: DIS HERA1+2 (green), dimuon SIDIS (purple), and charm SIDIS (orange). DIS HERA1+2 is distributed in a cross-shaped plane, charm SIDIS occupies the center of this cross, and dimuon SIDIS is a linear cluster crossing DIS HERA1+2. As the contribution of PC6 is increased, DIS HERA1+2 becomes almost singular in one direction (frame 5), indicating that this experiment has very little variability in the direction of PC6. The animation can be viewed at https://nspyrison.netlify.com/thesis/discluster\\_manualtour\\_pc6/."----

##DIS cluster, fig 8 of cook_dynamical_2018
load("./data/grDIScenter.rda")
load("./data/DIScluster_centered_basis.rda")
DIS_bas   <- DIScluster_centered_basis
DIS_dat   <- tourr::rescale(grDIScenter[, 1:6])
DIS_clas  <- factor(grDIScenter$disID)
DIS_mvar  <- 6
DIS_ang   <- .32
DIS_mtour <- manual_tour(basis = DIS_bas, manip_var = DIS_mvar, angle = DIS_ang)

## Figure
array2ggfacets(tour_array = DIS_mtour, data = DIS_dat,
               m_var = DIS_mvar, class = DIS_clas)

## Saving gif of radial tour for each PC; for the supplementary material
if (FALSE){
  for(i in 1:ncol(DIS_dat)){
    fn <- paste0("discluster_radialtour_pc", i, ".gif")
    play_manual_tour(basis = DIS_bas, data = DIS_dat, manip_var = i,
                     color = DIS_clas, shape = DIS_clas,
                     ggtheme = list(theme_void(), theme(legend.position = "none")),
                     fps = 8, start_pause = .5, end_pause = 1,
                     render_type = render_gganimate,
                     gif_filename = fn, 
                     gif_path = "./gifs")
  }
}

##M_VAR, RESULT
# 1, purple jet
# 2, *poor
# 3, black
# 4, plane
# 5, black and plane
# 6, best*purple and plane


## ----DISclusterBad, echo=F, warning=F, out.height='7in', out.width='5.83in',         fig.cap="Snapshots of a radial manual tour exploring the sensitivity PC2 to the structure of the DIS cluster, with color indicating experiment type: DIS HERA1+2 (green), dimuon SIDIS (purple), and charm SIDIS (orange). As contribution from PC2 is decreased, dimuon SIDIS becomes more distinguishable from the other two clusters (frames 10-14), indicating that in its absence PC2 is important. The animation can be viewed at https://nspyrison.netlify.com/thesis/discluster\\_manualtour\\_pc2/."----
##DIS cluster, fig 8 of cook_dynamical_2018
DIS_mvar  <- 2
DIS_mtour <- manual_tour(basis = DIS_bas, manip_var = DIS_mvar, angle = DIS_ang)

## Figure
array2ggfacets(tour_array = DIS_mtour, data = DIS_dat,
               m_var = DIS_mvar, class = DIS_clas)

## Saving gif for the supplementary material
if (FALSE){
  play_manual_tour(basis = DIS_bas, data = DIS_dat, manip_var = DIS_mvar,
                   color = DIS_clas, shape = DIS_clas,
                   ggtheme = list(theme_void(), theme(legend.position = "none")),
                   fps = 8, start_pause = .5, end_pause = 1,
                   render_type = render_gganimate,
                   gif_filename = "discluster_radialtour_pc2.gif", 
                   gif_path = "./gifs")
}

